
#include "hexfile.h"
#include "flash_operations.h"
#include "p33Fxxxx.h"
#include <string.h>

static unsigned short parse_hex_char(char c) {
  if( c >= '0' && c <= '9' )
    return c - '0';
  else if( c >= 'a' && c <= 'f' )
    return c - 'a' + 10;
  else if( c >= 'A' && c <= 'F' )
    return c - 'A' + 10;
  else
    return 0xFFFF;
}

static unsigned short parse_hex_byte(char* ptr) {
  unsigned short ret = parse_hex_char(ptr[0]);
  if( ret != 0xFFFF ) {
    ret <<= 4;
    ret |= parse_hex_char(ptr[1]);
  }
  return ret;
}

static unsigned char check_checksum(char* ptr, unsigned char num_bytes) {
  unsigned short checksum;
  unsigned char sum = 0;

  checksum = parse_hex_byte(ptr + num_bytes*2 + 6);

  ptr -= 2;
  num_bytes += 4;
  while( num_bytes-- ) {
    unsigned short val = parse_hex_byte(ptr);
    if( val > 255 )
      return 0;
    sum += val;
    ptr += 2;
  }
  sum = ~sum + 1;
  return sum == checksum;
}

static unsigned char process_hex(unsigned long address, char* ptr, unsigned char num_bytes, unsigned char do_write) {
  unsigned char i;

  address >>= 1; // hex file addresses are 32 bit per program word, native addresses are 16 bit per program word

  // ignore addresses out of range, they are probably configuration bits
  if( address >= 0x10000L )
	return 0;

  ptr += 6;
  num_bytes >>= 2;
  while( num_bytes ) {
	unsigned long word = 0;
    ptr += 8;
    for( i = 0; i < 4; ++i ) {
      ptr -= 2;
      word = (word<<8)|parse_hex_byte(ptr);
    }

#ifdef BOOTLOADER
	// ignore addresses that are within the bootloader
	if( address >= 4 && address < 0x9000 ) {
#else
	// ignore addresses that are outside the bootloader
	if( address >= 0x9000 ) {
#endif
	  if( do_write ) {
	    smart_write_flash_word(address, word);
      } else {
	    if( word != read_program_word(address) )
	      return 1;
      }
	}
    ptr += 8;
	address += 2;
    --num_bytes;
  }

  return 0;
}

static hex_result parse_hex_file(FIL* file, unsigned char do_write) {
  char buf[128];
  unsigned short buf_pos = 0, buf_len = 0, num_bytes;
  unsigned short address, record_type, temp;
  unsigned short upper_address = 0;
  unsigned char last_record_type = 255;
  hex_result ret = no_difference;

  while(1) {
    unsigned int read;
	if( f_read(file, (BYTE*)buf+buf_len, 128-buf_len, &read) != FR_OK )
	  return hex_file_read_error;
    buf_len += read;

    if( buf_pos == buf_len )
      return last_record_type == 1 ? ret : hex_file_invalid;
    else if( last_record_type == 1 )
      return hex_file_invalid;

    if( buf[buf_pos] != ':' && buf_pos + 3 >= buf_len )
      return hex_file_invalid;
    num_bytes = parse_hex_byte(buf+buf_pos+1);
    if( num_bytes > 16 )
      return hex_file_invalid;
    buf_pos += 3;
    if( (buf_pos + 8 + num_bytes*2 + 1 >= buf_len) )
      return hex_file_invalid;
    if( !check_checksum(buf+buf_pos, num_bytes) )
      return hex_file_invalid;
    record_type = parse_hex_byte(buf+buf_pos+4);
    if( record_type > 1 && record_type != 4 )
      return hex_file_invalid;
    address = parse_hex_byte(buf+buf_pos+2);
    if( address == 0xFFFF )
      return hex_file_invalid;
    temp = parse_hex_byte(buf+buf_pos+0);
    if( temp == 0xFFFF )
      return hex_file_invalid;
    address += temp<<8;
    if( record_type == 4 ) {
	  if( num_bytes != 2 || address != 0 )
	    return hex_file_invalid;
      upper_address = (((unsigned short)parse_hex_byte(buf+buf_pos+6))<<8)|parse_hex_byte(buf+buf_pos+8);
    } else if( record_type == 1 ) {
	  if( num_bytes != 0 || address != 0 )
	    return hex_file_invalid;
	} else {
      unsigned long full_address = (((unsigned long)upper_address)<<16)|address;
      if( num_bytes == 0 || (num_bytes&3) )
		return hex_file_invalid;
      if( process_hex(full_address, buf + buf_pos, num_bytes, do_write) )
        ret = difference;
    }
    buf_pos += 8 + num_bytes*2;
    if( buf[buf_pos] == '\r' )
      ++buf_pos;
    if( buf_pos >= buf_len )
      return hex_file_invalid;
    if( buf[buf_pos] == '\n' )
      ++buf_pos;
    else
      return hex_file_invalid;

    memmove(buf, buf+buf_pos, buf_len-buf_pos);
    buf_len -= buf_pos;
    buf_pos = 0;
    last_record_type = record_type;
  }
}


hex_result check_hex_file(FIL* file) {
	return parse_hex_file(file, 0);
}

unsigned char reflash_from_hex_file(FIL* file) {
	unsigned int old_IPL = disableInterrupts();
	parse_hex_file(file, 1);
	smart_write_flash_flush_buffer();
	enableInterrupts(old_IPL);
	return 0;
}
